import process from 'node:process'
import url from 'node:url'
import path from 'node:path'
import fs from 'node:fs'
import asyncFs from 'node:fs/promises'
import Koa from 'koa'
import KoaRouter from 'koa-router'
import sendFile from 'koa-sendfile'

const __filename = url.fileURLToPath(import.meta.url)
const __dirname = path.dirname(__filename)

const PORT = Number.parseInt(process.env.PORT, 10) || 3000
const app = new Koa()
const router = new KoaRouter()

//
// Serve HTML page containing the video player
//
router.get('/', async (ctx) => {
    await sendFile(ctx, path.resolve(__dirname, 'public', 'index.html'))

    if (!ctx.status) {
        ctx.throw(404)
    }
})

//
// Serve video streaming
//
router.get('/api/video/:name', async (ctx) => {
    //
    // Check video file name
    //
    const { name } = ctx.params

    if (
        !/^[a-z0-9-_ ]+\.(mp4|webm|ogg|ogv|mkv)$/i.test(name)
    ) {
        ctx.throw(400, 'Invalid video file name')
    }

    //
    // Check Range HTTP request header
    //
    const { request, response } = ctx
    const { range } = request.headers

    if (!range) {
        ctx.throw(400, 'Range not provided')
    }

    //
    // Check video file
    //
    const videoPath = path.resolve(__dirname, 'videos', name)

    try {
        await asyncFs.access(videoPath)
    } catch (err) {
        if (err.code === 'ENOENT') {
            ctx.throw(404)
        } else {
            ctx.throw(err.toString())
        }
    }

    //
    // Calculate start Content-Range
    //
    const parts = range.replace('bytes=', '').split('-')
    const startStr = parts[0] && parts[0].trim()
    const start = startStr ? Number.parseInt(startStr, 10) : 0

    //
    // Calculate video size and chunk size
    //
    const videoStat = await asyncFs.stat(videoPath)
    const videoSize = videoStat.size
    const chunkSize = 10 ** 6 // 1mb

    //
    // Calculate end Content-Range
    //
    const endStr = parts[1] && parts[1].trim()
    const rangeEnd = endStr ? Number.parseInt(endStr, 10) : undefined
    // Safari/iOS first sends a request with bytes=0-1 range HTTP header
    // probably to find out if the server supports byte ranges
    // We remove 1 byte because start and end start from 0
    const end = rangeEnd === 1
        ? rangeEnd
        : (Math.min(start + chunkSize, videoSize) - 1)
    // We add 1 byte because start and end start from 0
    const contentLength = end - start + 1

    //
    // Set HTTP response headers
    //
    response.set('Content-Range', `bytes ${start}-${end}/${videoSize}`)
    response.set('Accept-Ranges', 'bytes')
    response.set('Content-Length', contentLength)

    //
    // Send video file stream from start to end
    //
    const stream = fs.createReadStream(videoPath, { start, end })
    stream.on('error', (err) => {
        console.log(err.toString())
    })

    response.status = 206
    response.type = path.extname(name)
    response.body = stream
})

//
// We ignore ECONNRESET, ECANCELED and ECONNABORTED errors
// because when the browser closes the connection, the server
// tries to read the stream. So, the server says that it cannot
// read a closed stream.
//
app.on('error', (err) => {
    if (!['ECONNRESET', 'ECANCELED', 'ECONNABORTED'].includes(err.code)) {
        console.log(err.toString())
    }
})

//
// Add Koa Router middleware
//
app.use(router.routes())
app.use(router.allowedMethods())

//
// Start the server on the specified PORT
//
app.listen(PORT)
console.log('Video Streaming Server is running on Port', PORT)
