/**
 * Copyright © 2017 tesshu.com (webmaster@tesshu.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.tesshu.subsonic.client.sample4_music_andmovie;

import com.tesshu.subsonic.client.SuccessObserver;
import com.tesshu.subsonic.client.controller.Search2Controller;
import com.tesshu.subsonic.client.controller.StreamController;
import com.tesshu.subsonic.client.model.Child;
import com.tesshu.subsonic.client.model.MediaType;
import com.tesshu.subsonic.client.model.SearchResult2;
import com.tesshu.subsonic.client.util.IRequestUriObserver;

import org.apache.commons.lang3.builder.ToStringBuilder;
import org.apache.commons.lang3.builder.ToStringStyle;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.boot.SpringApplication;
import org.springframework.boot.autoconfigure.SpringBootApplication;
import org.springframework.boot.web.client.RestTemplateBuilder;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.ComponentScan;
import org.springframework.web.client.RestTemplate;

import javafx.application.Application;
import javafx.scene.Group;
import javafx.scene.Scene;
import javafx.scene.media.Media;
import javafx.scene.media.MediaPlayer;
import javafx.scene.media.MediaView;
import javafx.stage.Stage;

import java.util.stream.Collectors;

/**
 * Sample of argument pattern to play Stream. Please set estimateContentLength
 * to true.
 */
@SpringBootApplication
@ComponentScan(basePackages = { "com.tesshu.subsonic.client" })
public class StreamPlayMovieApplication extends Application {

  private static ApplicationContext                                context;

  private String                                                   format = "mp4";

  private String                                                   uriStr = null;
  
  private static final Log                                         LOG    = LogFactory
      .getLog(new Throwable().getStackTrace()[0].getClassName());

  public static void main(String[] args) {
    context = SpringApplication.run(StreamPlayMovieApplication.class, args);
    launch(args);
  }

  @Bean
  public RestTemplate restTemplate(RestTemplateBuilder builder) {
    return builder.build();
  }

  @Override
  public void start(Stage stage) throws Exception {

    Search2Controller search2 = context.getBean(Search2Controller.class);

    StreamController streamController = context.getBean(StreamController.class);

    SuccessObserver callback = context.getBean(SuccessObserver.class);

    SearchResult2 result2 = search2.get(
        "CORPSE BRIDE", // query, required = true
        0, // artistCount, required = false
        null, // artistOffset, required = false
        0, // albumCount, required = false
        null, // albumOffset, required = false
        10, // songCount, required = false
        null, // songOffset, required = false
        null // musicFolderId, required = false
    );

    Child movie = result2.getSongs().stream()
        .filter(child -> MediaType.VIDEO == child.getType())
        .filter(child -> format.equals(child.getSuffix()))
        .collect(Collectors.toSet()).iterator().next();

    LOG.info(ToStringBuilder.reflectionToString(movie,
        ToStringStyle.MULTI_LINE_STYLE));

    // not valid?(Perhaps, I have not done convert setting on the server side)
    @SuppressWarnings("unused")
    String size = Integer.toString(movie.getOriginalWidth()) + "x"
        + Integer.toString(movie.getOriginalHeight());
    
    final IRequestUriObserver uriCallBack = (subject, uri) -> {
      uriStr = uri.toString();
    };

    streamController.stream(

        movie, // id
        null, // maxBitRate
        format, // format
        null, // timeOffset
        null, // size
        true, // estimateContentLength
        false, // converted
        null, // streamCallback
        uriCallBack, callback);

    Group root = new Group();
    Scene scene = new Scene(root, movie.getOriginalWidth(),
        movie.getOriginalHeight());
    Media media = new Media(uriStr);
    media.errorProperty().addListener((observable, old, cur) -> {
      LOG.info(cur + " : " + uriStr);
    });

    MediaPlayer player = new MediaPlayer(media);
    player.statusProperty().addListener((observable, old, cur) -> {
      LOG.info(cur + " : " + uriStr);
    });

    MediaView view = new MediaView(player);
    ((Group) scene.getRoot()).getChildren().add(view);
    stage.setScene(scene);
    stage.show();
    player.play();

  }

  @Override
  public void stop() {
    SpringApplication.exit(context, () -> {
      return 0;
    });
    System.exit(0);
  }

}
