/**
 * Copyright © 2017 tesshu.com (webmaster@tesshu.com)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.tesshu.subsonic.client.sample2_service_customize;

import static java.util.Comparator.comparing;

import com.tesshu.subsonic.client.controller.GetIndexesController;
import com.tesshu.subsonic.client.controller.GetMusicFoldersController;
import com.tesshu.subsonic.client.model.Artist;
import com.tesshu.subsonic.client.model.Indexes;
import com.tesshu.subsonic.client.model.MusicFolder;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.springframework.boot.CommandLineRunner;
import org.springframework.boot.SpringApplication;
import org.springframework.boot.autoconfigure.SpringBootApplication;
import org.springframework.boot.web.client.RestTemplateBuilder;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.ComponentScan;
import org.springframework.web.client.RestTemplate;

import java.util.Optional;

import javax.annotation.Resource;

/**
 * Confirm connection to the server.
 */
@SpringBootApplication
@ComponentScan(basePackages = { "com.tesshu.subsonic.client" })
public class ReverseOrderOfArtistsApplication {

  private static final Log          LOG = LogFactory
      .getLog(new Throwable().getStackTrace()[0].getClassName());

  @Resource
  private GetMusicFoldersController getMusicFolders;

  @Resource
  private GetIndexesController      getIndexes;

  public static void main(String[] args) {
    SpringApplication.run(ReverseOrderOfArtistsApplication.class);
  }

  @Bean
  public RestTemplate restTemplate(RestTemplateBuilder builder) {
    return builder.build();
  }

  @Bean
  protected CommandLineRunner run(RestTemplate restTemplate) throws Exception {

    return args -> {

      // see ServiceCustomizer
      Optional<MusicFolder> musicFolder = getMusicFolders.getList().stream()
          .filter(folder -> "Music".equals(folder.getName())).findFirst();

      musicFolder.ifPresent(folder -> {
        Optional<Indexes> indexes = getIndexes.getOf(folder, null);
        indexes.ifPresent(i -> i.getIndexes().stream()
            .flatMap(index -> index.getArtists().stream())
            .sorted(comparing(Artist::getName).reversed())
            .forEach(artist -> LOG.info(artist)));
      });

    };
  }

}
